%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This program plots IRFs to a technology shock for the RBC model with
% variable labor supply.
%
% The Jacobian matrices for the linearized system are set up so that
% one identity is still present. The model's solution is obtained via 
% the qz decomposition.
%
% See class handout.
%
% Luca Guerrieri 1/30/2007
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear


%% Set parameter values
% Discount factor  (default .99)
betap = 0.99;

% Frisch Elasticity of labor supply (2/chip);
chip = 10;

% Steady state labor supply
hwp = 0.5;

% depreciation rate (default 0.025)
deltap = 0.025;

% elasticity of substitution between capital and labor
el = 0.5;
thetap = 1/el;
 
% Capital share
kpshrp = 0.4;

% Set nu so as to hit desired capital share
nup = (kpshrp*(betap * deltap - betap + 1)^(1/thetap-1)*(betap)^(1-1/thetap))^thetap;

% saving rate
sp = (nup/(deltap-1+1/betap))^(1/thetap)*(deltap);  

% parameter determining labor supply in SS
chi0p = (1-hwp)^(chip)*(1/(1-sp))*(1/hwp)*(1-nup*((sp/deltap)^(1-thetap)));

rhonp = 0.95;    
rhomp = 0.95;


%% compute the steady state values for the variables in the model

hwss = hwp;
gdpss = hwp*(1-nup*(sp/(deltap))^(1-thetap))^(1/(thetap-1));
ipss = (sp * gdpss);
ccss = (gdpss- ipss);
kpss = ipss/deltap;
rkss = (deltap - 1 + 1/betap);
rwss = (gdpss/hwss)^thetap;
lambdass = 1/ccss;
gammass = lambdass;
nss=0;
mss=0;

%% create the matrices that describe the linearized conditions for the model's equilibrium
%
%  These conditions take the form
%  psi x_{t+1} = phi x_{t} + f_{t+1} 
%

% constructing psi and phi, the ordering of the variables is as follows
varlist = char('m','n','kp','cc','gdp','hw','rk','rw')

% the ordering of the equations is:
% shock process for m
% shock process for n
% capital accumulation equation
% consumption euler equation
% resource constraint (GDP identity)
% foc for labor for households' problem
% foc for capital from firms' problem
% foc for labor from firms' problem

psi = [1 zeros(1,7)
       0 1 zeros(1,6)
       0 0 1 zeros(1,5)
       0 0 0 -betap/ccss^2*(1-deltap + rkss) 0 0 betap/ccss 0 
       zeros(4,8)];

   
deriv1=nup*thetap*(gdpss/kpss)^(thetap-1); 
deriv2=thetap*(gdpss/hwss)^(thetap-1);

phi = [rhomp zeros(1,7)
       0 rhonp zeros(1,6)
       0 0 (1-deltap) -1 1 0 0 0 
       zeros(1,3) -1/ccss^2 zeros(1,4)
       rkss*kpss rwss*hwss rkss 0 -1 rwss 0 0
       0 0 0 -rwss/ccss^2 0 -chip*chi0p*(1-hwss)^(-chip-1) 0 1/ccss
       rkss-deriv1*gdpss/kpss 0 -deriv1*gdpss/kpss^2 0 deriv1/kpss 0 -1 0
       0 rwss-deriv2*gdpss/hwss 0 0 deriv2/hwss -deriv2*gdpss/hwss^2 0 -1]

nvars = size(phi,1);
npredet = 3;
njump = nvars-npredet;

%% Obtain the Schur decomposition of phi and psi

% N.B.: the order of the arguments has meaning for the decomposition
% the LHS matrix psi, comes second.
[phiTr psiTr q z] = qz(phi,psi);

% Get the generalized eigenvalues of phi and psi
% using the output of the generalized complex Schur decomposition
E = ordeig(phiTr,psiTr);
% could also have done
% E = diag(phiTr)./diag(psiTr)


%% Check BK conditions
% we are looking for as many explosive eigenvalues as
% the number of non-predetermined jump variables in order
% to let the program continue. 
explpos = find(abs(E)>1)

if isempty(explpos)
    error('Not enough explosive roots')
elseif (length(explpos)>njump)
    error('Too many explosive roots, multiple RE equilibria')
elseif (length(explpos)<njump)
    error('Not enough explosive roots, no RE equilibrium')
end

%% Reorder the Schur decomposition output so that the stable eigenvalues come first.
stablepos = find(abs(E)<=1);

selector = zeros(nvars,1);
for indx=1:npredet
    selector(stablepos(indx))=1;
end
[phiTr psiTr q z]=ordqz(phiTr, psiTr,q,z,selector);

%% Create partitions (do not need all partitions)
psiTr11 = psiTr(1:npredet,1:npredet);
phiTr11 = phiTr(1:npredet,1:npredet);
q11 = q(1:npredet,1:npredet); q12 = q(1:npredet,npredet+1:nvars);
q21 = q(npredet+1:nvars,1:npredet); q22 = q(npredet+1:nvars,npredet+1:nvars);
z11 = z(1:npredet,1:npredet); 
z21 = z(npredet+1:nvars,1:npredet);

A = z11*psiTr11^-1*phiTr11*z11^-1
B = z11*psiTr11^-1*(q11-q12*(q22)^-1*q21)
C = z21*z11^-1


%% Create IRFs
% labor-augmenting shock
nperiods = 100;
history = zeros(nvars,nperiods+1);
epsilon = zeros(npredet,nperiods+1);
epsilon(2,2) = 0.01/0.606450705495247;
for t = 2:nperiods+1
    history(1:npredet,t) = A*history(1:npredet,t-1) + B*epsilon(:,t);
    history(npredet+1:nvars,t) = C*history(1:npredet,t);
end

for i=1:nvars
    eval(['f0_',varlist(i,:),'=history(i,2:end);'])
end

f0_ip = f0_gdp - f0_cc;

% capital augmenting shock
history = zeros(nvars,nperiods+1);
epsilon = zeros(npredet,nperiods+1);
epsilon(1,2) = 0.01/0.424185154949881;
for t = 2:nperiods+1
    history(1:npredet,t) = A*history(1:npredet,t-1) + B*epsilon(:,t);
    history(npredet+1:nvars,t) = C*history(1:npredet,t);
end

for i=1:nvars
    eval(['f1_',varlist(i,:),'=history(i,2:end);'])
end       
        
f1_ip = f1_gdp - f1_cc;

%% Make plots
figure
periods = 1:nperiods;

subplot(3,2,1)
plot(periods,f0_gdp/gdpss*100,'k',periods,f1_gdp/gdpss*100,'r--')
legend(char('Labor-Augmenting Shock','Capital-Augmenting Shock'))
title('Output')
ylabel('Percent')

subplot(3,2,2)
plot(periods,f0_kp/kpss*100,'k',periods,f1_kp/kpss*100,'r--')
title('Capital')
ylabel('Percent')

subplot(3,2,3)
plot(periods,f0_ip/ipss*100,'k',periods,f1_ip/ipss*100,'r--')
title('Investment')
ylabel('Percent')

subplot(3,2,4)
plot(periods,f0_cc/ccss*100,'k',periods,f1_cc/ccss*100,'r--')
title('Consumption')
ylabel('Percent')

subplot(3,2,5)
plot(periods,f0_n*100,'k',periods,f1_m*100,'r--')
title('Productivity')
ylabel('Percent')

subplot(3,2,6)
plot(periods,f0_hw/hwss*100,'k',periods,f1_hw/hwss*100,'r--')
title('Hours Worked')
ylabel('Percent')
