%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This program plots IRFs to a technology shock for the RBC model with
% fixed labor supply.
%
% See class handout.
%
% Luca Guerrieri 1/24/2007
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%% set the parameter values
deltap = 0.025;
betap = 0.99;
alphap = 0.33;
rhop = 0.95;

%% compute the steady state values for the variables in the model
zss = 0;
kss = (1/alphap*(1/betap-1+deltap))^(1/(alphap-1));
yss = kss^(alphap);
css = kss^alphap-deltap*kss;
iss = deltap*kss; 
lambdass = 1/css;
gammass = 1/css;

%% create the matrices that describe the linearized conditions for the model's equilibrium
%
%  These conditions take the form
%  psi x_{t+1} = phi x_{t} + f_{t+1}
%
psi =[-betap/css^2*(1-deltap+alphap*kss^(alphap-1)) betap/css*alphap*(alphap-1)*kss^(alphap-2) betap/css*alphap*kss^(alphap-1)
0 1 0
0 0 1];

phi = [ -1/css^2 0 0 
-1 (1-deltap)+alphap*kss^(alphap-1) kss^alphap
0 0 rhop];

%% find the model's solution

A = psi^(-1)*phi

% next find the eigenvectors of A, stack them columnwise in V
% and find the eigenvalues of A, arrange them on the diagonal of D
% N.B.: Matlab syntax is case sensitive. If you capitalize a variable name
% stick with it.
[V,D] = eig(A)

% extract the eigenvalues from the matrix D and put them in a vector d.
d = diag(D)

% Find the position within the vector d that corresponds to the explosive
% eigenvalue and store it in i
i = find(abs(d)>1)

% Check Blanchard Kahn Conditions (we are looking for 1 explosive
% eigenvalue)
if isempty(i)
    error('There are two many explosive eigenvalues')
elseif size(i)>1
    error('There are two few explosive eigenvalues')
end

% Define c_k and c_z that allow us to express c_t as a combination of k_t
% and z_t
Vinv = V^-1;

ck = - Vinv(i,2)/Vinv(i,1);

cz = - Vinv(i,3)/Vinv(i,1);

%% Define the decision rule for k_t and z_t.
%
%  k_{t+1}            k_t
%            =  B            +   C   epsilon_t+1
%  z_{t+1}            z_t  

B = A(2:3,1)*[ck cz] + A(2:3,2:3); 

C=[0; 1];

%% Generate IRFs for k and z
nperiods = 100;

history = zeros(2,nperiods+1);
epsilon = zeros(1,nperiods+1);
epsilon(2) = 0.01;
for t = 2:nperiods+1
    history(:,t) = B*history(:,t-1) + C*epsilon(t);
end

k = history(1,2:end);
z = history(2,2:end);

%% Back out the responses of consumption, investment, and output
c = [ck cz]*history(:,2:end);
y = alphap*kss^(alphap-1)*k + kss^alphap*z;
ii =y-c;
saving = 1/yss*ii - iss/yss^2*y;


%% Make plots
figure

subplot(3,2,1)
plot(y/yss*100)
title('Output')
ylabel('Percent')

subplot(3,2,2)
plot(k/kss*100)
title('Capital')
ylabel('Percent')

subplot(3,2,3)
plot(ii/iss*100)
title('Investment')
ylabel('Percent')

subplot(3,2,4)
plot(c/css*100)
title('Consumption')
ylabel('Percent')

subplot(3,2,5)
plot(z*100)
title('Productivity')
ylabel('Percent')

subplot(3,2,6)
plot(saving*100)
title('Saving')
ylabel('Percentage Point')