%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This program plots IRFs to a technology shock for the RBC model with
% fixed labor supply.
%
% The Jacobian matrices for the linearized system are set up so that
% one identity is still present. The model's solution is obtained via 
% the qz decomposition.
%
% See class handout.
%
% Luca Guerrieri 1/30/2007
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear

%% set the parameter values
deltap = 0.025;
betap = 0.99;
alphap = 0.33;
rhop = 0.95;

%% compute the steady state values for the variables in the model
zss = 0;
kss = (1/alphap*(1/betap-1+deltap))^(1/(alphap-1));
yss = kss^(alphap);
css = kss^alphap-deltap*kss;
iss = deltap*kss; 
lambdass = 1/css;
gammass = 1/css;

%% create the matrices that describe the linearized conditions for the model's equilibrium
%
%  These conditions take the form
%  psi x_{t+1} = phi x_{t} + f_{t+1} 
%
psi =[1 0 0 0 
    0 1 0 0  
betap/css*alphap*kss^(alphap-1)  betap/css*alphap*(alphap-1)*kss^(alphap-2) -betap/css^2*(1-deltap+alphap*kss^(alphap-1)) 0 
0 0 0 0];

phi = [ rhop 0 0 0  
0 (1-deltap) 0 1  
0 0 -1/css^2 0 
-kss^alphap -alphap*kss^(alphap-1) 1 1];

nvars = size(phi,1);
npredet = 2;
njump = nvars-npredet;

%% Obtain the Schur decomposition of phi and psi

% N.B.: the order of the arguments has meaning for the decomposition
% the LHS matrix psi, comes second.
[phiTr psiTr q z] = qz(phi,psi);

% Get the generalized eigenvalues of phi and psi
% using the output of the generalized complex Schur decomposition
E = ordeig(phiTr,psiTr);
% could also have done
% E = diag(phiTr)./diag(psiTr)


%% Check BK conditions
% we are looking for as many explosive eigenvalues as
% the number of non-predetermined jump variables in order
% to let the program continue. 
expl_pos = find(abs(E)>1)

if isempty(expl_pos)
    error('Not enough explosive roots')
elseif (size(expl_pos)>njump)
    error('Too many explosive roots, multiple RE equilibria')
elseif (size(expl_pos)<njump)
    error('Not enough explosive roots, no RE equilibrium')
end

%% Reorder the Schur decomposition output so that the stable eigenvalues come first.
stable_pos = find(abs(E)<=1)

selector = zeros(nvars,1)
for indx=1:npredet
    selector(stable_pos(indx))=1;
end
[phiTr psiTr q z]=ordqz(phiTr, psiTr,q,z,selector);

%% Create partitions (do not need all partitions)
psiTr11 = psiTr(1:npredet,1:npredet);
phiTr11 = phiTr(1:npredet,1:npredet);
q11 = q(1:npredet,1:npredet); q12 = q(1:npredet,npredet+1:nvars);
q21 = q(npredet+1:nvars,1:npredet); q22 = q(npredet+1:nvars,npredet+1:nvars);
z11 = z(1:npredet,1:npredet); 
z21 = z(npredet+1:nvars,1:npredet);

A = z11*psiTr11^-1*phiTr11*z11^-1
B = z11*psiTr11^-1*(q11-q12*(q22)^-1*q21)
C = z21*z11^-1


%% Create IRFs
nperiods = 100;
history = zeros(nvars,nperiods+1);
epsilon = zeros(2,nperiods+1);
epsilon(1,2) = 0.01;
for t = 2:nperiods+1
    history(1:npredet,t) = A*history(1:npredet,t-1) + B*epsilon(:,t);
    history(npredet+1:nvars,t) = C*history(1:npredet,t);
end

z = history(1,2:end);
k = history(2,2:end);
c = history(3,2:end);
ii = history(4,2:end);
y = c+ii;
saving = 1/yss*ii - iss/yss^2*y;



%% Make plots
figure

subplot(3,2,1)
plot(y/yss*100)
title('Output')
ylabel('Percent')

subplot(3,2,2)
plot(k/kss*100)
title('Capital')
ylabel('Percent')

subplot(3,2,3)
plot(ii/iss*100)
title('Investment')
ylabel('Percent')

subplot(3,2,4)
plot(c/css*100)
title('Consumption')
ylabel('Percent')

subplot(3,2,5)
plot(z*100)
title('Productivity')
ylabel('Percent')

